#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/slab.h>

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <linux/mutex.h>
#include <linux/kthread.h>
#include <asm/uaccess.h>
#include "m24m01.h"
#include "CRC.h"
#include "../../../external/crestron/ERInterfaceDLL/nvram_dataMap.h"


//-------------------------------------------------------------------------
// Preprocessor defines.
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#define DEBUG 0
// Feature: Fail to read/return data if it might be invalid, based on CRC.
#define NVRDS_READ_FAIL_ON_CRC_ERR (0)

// Feature: Fail to write/store data if it might be invalid, based on CRC.
#define NVRDS_WRITE_FAIL_ON_CRC_ERR (0)

// For test/debug: Simulate data from a new/unused EEPROM (all bytes 0xFF).
#define NVRAM_SIMULATE_NEW_EEPROM (0)


#define DRIVER_NAME	"m24m01"
// Characteristics of the NVRAM h/w in use, which is a serial EEPROM chip.

// Number of bytes in each serial EEPROM page.
#define NVRAM_PAGE_SIZE (256)

// Total number of serial EEPROM pages.
#define NVRAM_PAGE_COUNT (512)

// Serial EEPROM capacity, in bytes.
#define NVRAM_CAPACITY (NVRAM_PAGE_SIZE * NVRAM_PAGE_COUNT)


#if DEBUG //increases wear leveling frequency
#define PAGE_WRITES_BEFORE_UPDATE (30)
#define WEAR_CHECK_INTERVAL (128)
#else
#define PAGE_WRITES_BEFORE_UPDATE (300)
#define WEAR_CHECK_INTERVAL (8192)
#endif
typedef u32 SignatureType;

struct mutex fs_mutex;

struct MemoryPageRangeInfo
{
    size_t firstIndex;  // Index of first NVRAM page.
    size_t count;       // Number of NVRAM pages.
    size_t firstLength; // Length of first NVRAM page.
    size_t lastLength;  // Length of last NVRAM page.
};

// Format of payload of an I2C packet:  A "dummy write" is part of a read
//  operation.  It specifies the NVRAM address where to begin reading data.
struct i2cPacketPayload_DummyWrite
{
    u8 memAddrHi;
    u8 memAddrLo;
};

// Format of payload of an I2C packet:  To write a whole page of NVRAM data.
struct i2cPacketPayload_PageWrite
{
	u8 memAddrHi;
	u8 memAddrLo;
	u8 memWord [NVRAM_PAGE_SIZE];
};

struct m24m01_eeprom {
        struct miscdevice *misc;
        struct i2c_client *client;
        char usrBuf[PAGE_SIZE];
};

static bool UpdateStoredCrc (void);
static bool InvalidateStoredCrc (void);
static bool StoreAsCrc (CrcValueType const crc);
static bool CalculateCrc (CrcValueType* crcOut);

static bool isOutsideSignature (size_t const sectOffset, size_t const length);
static bool isOutsideStoredCrc (size_t const sectOffset, size_t const length);
static bool isOutsideCrcRawData (size_t const sectOffset, size_t const length);
static bool isOutsideWearTables (size_t const sectOffset, size_t const length);


static void WearLevelingThread( struct work_struct *work);
static bool IsWearTablesInitialized(void);
static bool InitializeWearTables(void);
static bool LoadWearTables(void);
static bool ValidateWearTables(void);
static bool SaveWearTables(void);
static bool Swap(size_t PhysicalPage_1, size_t PhysicalPage_2);
static size_t PhysicalToLogical(size_t const PhysicalOffset);
static size_t LogicalToPhysical(size_t const LogicalOffset);
static void CreateLogicalToPhysicalTable(void);

static size_t nvram_len;

// Baseline I2C address of Serial EEPROM chip.
//  - NOTE: Only a starting point for defining which I2C addresses to use.
static u8 const i2cAddr = 0xA8 >> 1;
// I2C address for lower half of Serial EEPROM memory.
static u8 const i2cAddr_MemLo = (0xA8 >> 1) | 0; // Lowest bit ("P0") is clear.

// I2C address for upper half of Serial EEPROM memory.
static u8 const i2cAddr_MemHi = (0xA8 >> 1) | 1; // Lowest bit ("P0") is set.

// ...Boundary between lower & upper halves of Serial EEPROM memory.
//  - Next item defines the start of the upper half of Serial EEPROM memory.
//  - The lower half runs from address zero to the start of the upper half.
static size_t const memAddrStartHi = NVRAM_CAPACITY / 2;

static u8* shadowPtr; // Start of shadow RAM.

static struct m24m01_eeprom *gEEprom = NULL;

static bool isCrcValid;
WEAR_LEVELING_TABLES_STRUCT WearTables; //local copy of wear tables in nvram
size_t LogicalToPhysicalTable[WL_WEAR_TABLE_NUMBER_ENTRIES]; //logical to physical page mapping
bool swapFlags[WL_WEAR_TABLE_NUMBER_ENTRIES];//list of swap candidates
struct mutex nvramMutex;
static struct workqueue_struct *wearLeveling_wq;
static struct work_struct wearLeveling_work;

DECLARE_COMPLETION(init_completed);

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Thread to handle wear leveling in the background.
* \note     a single event may trigger multiple swaps, and always swaps to page with least total writes found.
* \param    lpParam - 'this' pointer.
* \return   Thread shouldnt exit unless failure occurred
*/
void WearLevelingThread( struct work_struct *work )
{
	size_t i,j;

//	printk("%s: \r\n", __FUNCTION__);

	//check list of swap candidates
	for(j = 0; j < NVRAM_CAPACITY/NVRAM_PAGE_SIZE; j++)
	{
		if(swapFlags[j])
		{
			//find min to swap
			unsigned long min = 0x7FFFFFFF;
			ssize_t physicalIndexMin = -1;
			mutex_lock(&nvramMutex);
			for(i = 0; i < NVRAM_CAPACITY/NVRAM_PAGE_SIZE; i++)
			{
				//only allow swap of entries not in wear leveling tables
				if(i < WEAR_LEVELING_TABLES/NVRAM_PAGE_SIZE ||
					i >= (WEAR_LEVELING_TABLES+WEAR_LEVELING_TABLES_SIZE)/NVRAM_PAGE_SIZE)
				{
					if(WearTables.TotalWrites[i] < min)
					{
						min = WearTables.TotalWrites[i];
						physicalIndexMin = i;
					}
				}
			}
			if(physicalIndexMin != -1)
			{
				Swap(j, physicalIndexMin);
				swapFlags[j] = false;
			}
			mutex_unlock(&nvramMutex);
		}
	}
	return;
}

static int ReadPage (size_t const memPageIndex, u8 *const pageOutBuf)
{
     bool retval = 0; // ASSUME negative outcome.
     struct i2cPacketPayload_DummyWrite dummyWriteMsg = {};
     struct i2c_msg msgs[2];
#if DEBUG
     size_t wordIndex;//debug
#endif

//     printk(KERN_ERR "ReadPage %d\n",memPageIndex);

    // Verify parameter(s).
    if ((memPageIndex < NVRAM_PAGE_COUNT) &&
        (NULL != pageOutBuf))
    {
        // Memory address at which to begin reading.
        size_t const memAddrStart = memPageIndex * NVRAM_PAGE_SIZE;

        // Random read of memory must begin with a dummy write packet.

        dummyWriteMsg.memAddrLo = memAddrStart & 0xFF;
        dummyWriteMsg.memAddrHi = (memAddrStart >> 8) & 0xFF;

        gEEprom->client->addr = ((memAddrStart < memAddrStartHi) ? i2cAddr_MemLo : i2cAddr_MemHi);
    	msgs[0].addr = gEEprom->client->addr;
    	msgs[0].flags = 0;
    	msgs[0].len = sizeof(dummyWriteMsg);
    	msgs[0].buf = (u8*)&dummyWriteMsg;

    	msgs[1].addr = gEEprom->client->addr;
    	msgs[1].flags = I2C_M_RD;
    	msgs[1].len = NVRAM_PAGE_SIZE;
    	msgs[1].buf = pageOutBuf;

    	retval = i2c_transfer(gEEprom->client->adapter, msgs, 2);
    	msleep(1);
    	if(retval < 0)
    	{
    		printk("i2c_transfer error: %d\n",retval);
    		return retval;
    	}
#if DEBUG
    	for (wordIndex = 0; wordIndex < /*_countof(seqReadMsg)*/ NVRAM_PAGE_SIZE; wordIndex += 16)
		{
			printk("[%02X] %02X %02X %02X %02X %02X %02X %02X %02X-%02X %02X %02X %02X %02X %02X %02X %02X.\r\n"
				, memAddrStart + wordIndex
				, pageOutBuf[wordIndex + 0x0]
				, pageOutBuf[wordIndex + 0x1]
				, pageOutBuf[wordIndex + 0x2]
				, pageOutBuf[wordIndex + 0x3]
				, pageOutBuf[wordIndex + 0x4]
				, pageOutBuf[wordIndex + 0x5]
				, pageOutBuf[wordIndex + 0x6]
				, pageOutBuf[wordIndex + 0x7]
				, pageOutBuf[wordIndex + 0x8]
				, pageOutBuf[wordIndex + 0x9]
				, pageOutBuf[wordIndex + 0xA]
				, pageOutBuf[wordIndex + 0xB]
				, pageOutBuf[wordIndex + 0xC]
				, pageOutBuf[wordIndex + 0xD]
				, pageOutBuf[wordIndex + 0xE]
				, pageOutBuf[wordIndex + 0xF]
				);
		}
#endif
    }
    return retval;
}

int ReadCurrentWord (void)
{
     int retval = 0; // ASSUME negative outcome.

    // Prepare to read the current memory word.
    u8 memWord = 0x77;

    // I2C packet for read operation.
    struct i2c_msg packet = {};
    packet.addr = i2cAddr;
    packet.flags = I2C_M_RD;
    packet.buf = &memWord;
    packet.len = sizeof(memWord);

    // Perform the I2C transfer.
    retval = i2c_transfer(gEEprom->client->adapter, &packet, 1);

    // Show results of I2C transfer.
    return retval;
}

static bool WritePage (size_t const memPageIndex, u8 const *const pageInBuf)
{
    bool retval = false; // ASSUME negative outcome.
    int result;
    struct i2c_msg msgs;
    struct i2cPacketPayload_PageWrite pageWriteMsg = {};
    struct timeval current_tv, start_tv;
    int elapsedTime;

//    printk(KERN_ERR "WritePage: %x\n",memPageIndex);
    // Verify parameter(s).
    if ((memPageIndex < NVRAM_PAGE_COUNT) &&
        (NULL != pageInBuf))
    {
        // Memory address at which to begin reading.
        size_t const memAddrStart = memPageIndex * NVRAM_PAGE_SIZE;
        pageWriteMsg.memAddrLo = memAddrStart & 0xFF;
        pageWriteMsg.memAddrHi = (memAddrStart >> 8) & 0xFF;

		memcpy(pageWriteMsg.memWord, pageInBuf, NVRAM_PAGE_SIZE);

		msgs.addr = ((memAddrStart < memAddrStartHi) ? i2cAddr_MemLo : i2cAddr_MemHi);
		msgs.flags = 0;
		msgs.len = sizeof(pageWriteMsg);
		msgs.buf = (u8*)&pageWriteMsg;

		result = i2c_transfer(gEEprom->client->adapter, &msgs, 1);

		// Avoid problem:  The memory h/w device (a serial EEPROM chip) is
		//  now busy with its internal write cycle, storing the given data.
		//  During that time, the memory device ignores I2C packets:  They
		//  would fail due to lack of acknowledgment.
		// ...Wait until the memory device responds to I2C packets again.
		do_gettimeofday(&start_tv);
//		printk("t: \r\n");//,start_tv.tv_sec,start_tv.tv_usec);
		while (retval == false)
		{
			do_gettimeofday(&current_tv);
			result = ReadCurrentWord();
			// Check for a simple response from the memory h/w.
			if (result > 0)
			{
				// Got desired outcome:
				//  - Prepare to exit the wait-loop.
				//  - Prepare to report overall success.
				retval = true;
			}
			else // Check whether worst-case time-out occurred.
			{
				elapsedTime = (current_tv.tv_sec - start_tv.tv_sec) * 1000;
				elapsedTime += (current_tv.tv_usec - start_tv.tv_usec) / 1000;
//               	printk(" t: \r\n");//,current_tv.tv_sec,current_tv.tv_usec);
				if (unlikely(result == -EINTR))
				{
					// Wait failed due process interrupted.
					printk("%s: FAILED after write: interrupted.\r\n",
						__FUNCTION__);

					// Exit the wait-loop.  Already assumed negative outcome.
					break;
				}
				else if (elapsedTime > 100)
				{
					// Wait failed due to time-out.
					printk("%s: FAILED after write: Timed-out. %d\r\n",
						__FUNCTION__,result);

					// Exit the wait-loop.  Already assumed negative outcome.
					break;
				}
				else
				{
					msleep(1);
				}
			}
		}
    }
    else
    {
        printk("%s: BAD parameter: memPageIndex = %u | pageInBuf = %p.\r\n",
            __FUNCTION__, memPageIndex, pageInBuf);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return retval;
}

bool IsValidRange (size_t const offset, size_t const length, bool const allowZeroLength)
{
    bool result = 0; // ASSUME negative outcome.

//    printk(KERN_ERR "IsValidRange\n");
    //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // NOTE:  The comparison and arithmetic operations below are designed
    //  to avoid numeric overflow/underflow.  Specifically, they avoid
    //  subtracting a larger unsigned integer value from a smaller one.
    //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    // Verify parameter: The starting point must be within the range.
    if (offset < NVRAM_CAPACITY)
    {
        // Determine the distance from the given starting point to the end
        //  of the range.  It is the maximum allowed length of the range.
        size_t const maxLength = NVRAM_CAPACITY - offset;

        // Verify parameter: Range-check the length of the given range.
        if ((allowZeroLength || (0 < length)) &&
            (maxLength >= length))
        {
            // Prepare to report success.
            result = 1;
        }
    }
    return result;
}

bool CalcAssociatedPages (size_t const offset, size_t const length, struct MemoryPageRangeInfo *pageInfo)
{
    bool result = false; // ASSUME failure.

//    printk(KERN_ERR "CalcAssociatedPages\n");
    // A length of zero is a special case: No NVRAM pages are affected.
    if (0 == length)
    {
        // Verify remaining parameter(s).
        if (NVRAM_CAPACITY > offset)
        {
            // Prepare to report results of do-nothing operation.
            pageInfo->firstIndex = 0;
            pageInfo->count = 0;
            pageInfo->firstLength = 0;
            pageInfo->lastLength = 0;
            result = true;
        }
    }

    else if (IsValidRange(offset, length, false)) // Verify parameter(s).
    {
        // Explicitly use offsets from the underlying range.
        //  - Conversion is moot when "underlying" range starts at zero.
        size_t const offsetStart = offset;

        // Determine the end of the affected range of NVRAM offsets.
        //
        //  - NOTE: 'length > 0' within the current 'if' block.
        //          Therefore, it is safe to subtract 1 from length value.
        //          Otherwise, length/offset values could wrap below zero.
        //
        size_t const offsetEnd = offsetStart + length; // NOTE: Past final byte.
        size_t const offsetFinal = offsetEnd - 1; // Final affected byte.

        // Determine the affected range of NVRAM pages.
        size_t const pageIndexBegin = offsetStart / NVRAM_PAGE_SIZE;
        size_t const pageIndexFinal = offsetFinal / NVRAM_PAGE_SIZE;

        //- - - - -
        // The first and last affected NVRAM pages can be treated as partial
        //  pages.  For example, the range of affected bytes might begin in
        //  the middle of the first page and end in the middle of the last
        //  page (Any pages between the first and last one would be affected
        //  in whole).  In order to track how many bytes are relevant, it is
        //  necessary to determine how many are affected in the first page
        //  and in the last page.  Such calculations are done below, and
        //  they depend on whether the first and last pages are different.
        //- - - - -

        // Affected length of first affected NVRAM page.
        size_t const affectedLenBegin = (pageIndexBegin != pageIndexFinal)
            ? (NVRAM_PAGE_SIZE - (offsetStart % NVRAM_PAGE_SIZE))
            : length; // Whole length is covered by a single NVRAM page.

        // Affected length of final affected NVRAM page.
        size_t const affectedLenFinal = (pageIndexBegin != pageIndexFinal)
            ? ((offsetFinal % NVRAM_PAGE_SIZE) + 1)
            : 0; // Whole length is already counted as part of first page.

        // Prepare to report results.
        pageInfo->firstIndex = pageIndexBegin;
        pageInfo->count = (pageIndexFinal - pageIndexBegin) + 1;
        pageInfo->firstLength = affectedLenBegin;
        pageInfo->lastLength = affectedLenFinal;
        result = true;
    }
//    printk(KERN_ERR "CalcAssociatedPages %d %d %d %d\n",pageInfo->firstIndex,pageInfo->count,pageInfo->firstLength,pageInfo->lastLength);
    return result;
}

ssize_t ReadShadow (size_t const offset, void* const pBuffer, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.

    // Verify parameter(s).
    if ((NULL != pBuffer) &&
        IsValidRange(offset, length, true))
    {
        // Copy data from shadow RAM to the output buffer.
        memcpy(pBuffer, shadowPtr + offset, length);
        result = length; // Prepare to report success.
    }
    else
    {
        printk("%s: BAD parameter: offset %u | pBuffer %p | length %u.\r\n",
            __FUNCTION__, offset, pBuffer, length);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return result;
}

ssize_t WriteShadow (size_t const offset, const void* const pBuffer, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.

//    printk(KERN_ERR "WriteShadow: %x\n",offset);
    // Verify parameter(s).
    if ((NULL != pBuffer) &&
        IsValidRange(offset, length, true))
    {
        // Copy data from the input buffer to shadow RAM.
        memcpy(shadowPtr + offset, pBuffer, length);
        result = length; // Prepare to report success.
    }
    else
    {
    	printk("%s: BAD parameter: offset %u | pBuffer %p | length %u.\r\n",
    	            __FUNCTION__, offset, pBuffer, length);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return result;
}

int CompareShadow (size_t const offset, const void* const pBuffer, size_t const length)
{
	int result = -1; // ASSUME different.

    // Verify parameter(s).
    if ((NULL != pBuffer) &&
        IsValidRange(offset, length, true))
    {
        // Copy data from shadow RAM to the output buffer.
        result = memcmp(pBuffer, shadowPtr + offset, length);
    }
    else
    {
        printk("%s: BAD parameter: offset %u | pBuffer %p | length %u.\r\n",
            __FUNCTION__, offset, pBuffer, length);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return result;
}

/**
* \author   Arto Kiremitdjian
* \date     12/18/2011
* \brief    Fills/writes a region of shadow RAM with a given data value.
* \detail   Takes an "all or nothing" approach on how much data to write.
* \param    offset - Offset in shadow RAM: Where to start the operation.
* \param    value - Input data: What value to write.
* \param    length - How many bytes to write, Length of the region to fill.
* \return   Number of bytes actually written.
* \retval   -1 = Failure.
*/
ssize_t FillShadow (size_t const offset, u8 const value, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.

    // Verify parameter(s).
    if (IsValidRange(offset, length, true))
    {
        // Fill the specified part of shadow RAM with the given data value.
        memset(shadowPtr + offset, value, length);
        result = length; // Prepare to report success.
    }
    else
    {
        printk("%s: BAD parameter: offset %u | length %u.\r\n",
            __FUNCTION__, offset, length);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return result;
}

ssize_t ReadAffectedPages (size_t const offset, size_t const length)
{
	ssize_t result = (-1); // ASSUME failure.

    // Determine which NVRAM pages are affected.
    struct MemoryPageRangeInfo pageInfo;
    size_t pageIndex;

//	printk(KERN_ERR "ReadAffectedPages\n");

    if (CalcAssociatedPages(offset, length, &pageInfo))
    {
        // Alias for the end of the NVRAM page range.  NOTE: Past last page.
        size_t const pageIndexEnd = pageInfo.firstIndex + pageInfo.count;

        // Traverse affected pages: Copy each from actual NVRAM to shadow RAM.
        ssize_t lengthRead = 0; // ASSUME failure, Update at each iteration.
        for (pageIndex = pageInfo.firstIndex; pageIndex < pageIndexEnd; ++pageIndex)
        {
            // Copy current page from actual NVRAM to shadow RAM.
            if (ReadPage(pageIndex, shadowPtr + (pageIndex * NVRAM_PAGE_SIZE)) >= 0)
            {
                // Track/accumulate how much has been read so far.
                if (pageInfo.firstIndex == pageIndex)
                {
                    lengthRead += pageInfo.firstLength;
                }
                else if (pageIndexEnd == (pageIndex + 1))
                {
                    lengthRead += pageInfo.lastLength;
                }
                else
                {
                    lengthRead += NVRAM_PAGE_SIZE;
                }
            }
            else
            {
                // Exit enclosing loop early, upon first failed NVRAM write.
                break;
            }
        }
        // Prepare to report result.
        result = lengthRead;
    }
    return result;
}

ssize_t WriteAffectedPages (size_t const offset, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.
	size_t pageIndex;

	// Determine which NVRAM pages are affected.
    struct MemoryPageRangeInfo pageInfo;
//    printk(KERN_ERR "WriteAffectedPages: %d %d\n",offset, length);
    if (CalcAssociatedPages(offset, length, &pageInfo))
    {
        // Alias for the end of the NVRAM page range.  NOTE: Past last page.
        size_t const pageIndexEnd = pageInfo.firstIndex + pageInfo.count;

        // Traverse affected pages: Copy each from shadow RAM to actual NVRAM.
        ssize_t lengthWritten = 0; // ASSUME failure, Update at each iteration.
        for (pageIndex = pageInfo.firstIndex; pageIndex < pageIndexEnd; ++pageIndex)
        {
            // Copy current page from shadow RAM to actual NVRAM.
            if (WritePage(pageIndex, shadowPtr + (pageIndex * NVRAM_PAGE_SIZE)))
            {
                // Track/accumulate how much has been written so far.
                if (pageInfo.firstIndex == pageIndex)
                {
                    lengthWritten += pageInfo.firstLength;
                }
                else if (pageIndexEnd == (pageIndex + 1))
                {
                    lengthWritten += pageInfo.lastLength;
                }
                else
                {
                    lengthWritten += NVRAM_PAGE_SIZE;
                }
            }
            else
            {
                printk("%s: FAILED to write page %u to NVRAM.\r\n",
                		__FUNCTION__, pageIndex);

                // Exit enclosing loop early, upon first failed NVRAM write.
                break;
            }
        }
        // Prepare to report result.
        result = lengthWritten;
    }
    return result;
}

/**
* \author   Arto Kiremitdjian
* \date     1/31/2012
* \brief    Allows the caller to read data directly from shadow RAM.
* \detail   Takes an "all or nothing" approach on how much data to read.
* \warning  Caller MUST NOT read past the originally-requested length.
*           Caller MUST NOT use the returned pointer to write any data.
*           Caller MUST NOT expose the returned pointer outside this driver.
* \param    offset - Offset in shadow RAM: Where to start the operation.
* \param    length - How much data to read.
* \return   Pointer to the desired data in shadow RAM, only for reading.
* \retval   NULL = Failure.
*/
u8 const* PeekShadow (size_t const offset, size_t const length)
{
    u8 const* result = NULL; // ASSUME failure.

    // Verify parameter(s).
    if (IsValidRange(offset, length, true))
    {
        result = shadowPtr + offset; // Prepare to report success.
    }
    else
    {
        printk("%s: BAD parameter: offset %u | length %u.\r\n",
            __FUNCTION__, offset, length);
        // Intentionally do nothing else: Already assumed negative outcome.
    }
    return result;
}

ssize_t Write (size_t const sectOffset, const void* const pBuffer, size_t const length,
                         bool const enforceWriteProtection)
{
	ssize_t result = -1; // ASSUME failure.
	// Write the given data to shadow RAM.
	if (IsValidRange(sectOffset, length, true))
	{
		if(CompareShadow(sectOffset, pBuffer, length) == 0)
			result = length;
		else
		{
			ssize_t const lengthWritten = WriteShadow(sectOffset, pBuffer, length);
			if (0 <= lengthWritten)
			{
	//			 printk("%s: %d %d %d\r\n", __FUNCTION__, sectOffset, length, lengthWritten);
				// Write/commit the affected pages to NVRAM.
				result = WriteAffectedPages(sectOffset, lengthWritten);
			}
		}
	}
	return result;
}

/**
* \author   Arto Kiremitdjian.
* \brief    Fills/writes a region of NVRAM with a given data value.
* \date     12/18/2011
* \param    offset - Where to start writing: NVRAM offset.
* \param    value - Input data: What value to write to NVRAM.
* \param    length - How many bytes to write, Length of the region to fill.
* \return   Number of bytes actually written.
* \retval   -1 = Failure.
*/
ssize_t Fill (size_t const offset, u8 const value, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.
    // Write the given data to shadow RAM.
    ssize_t const lengthWritten = FillShadow(offset, value, length);
    if (0 <= lengthWritten)
    {
        // Check whether to perform DEBUG/TEST simulation or normal operation.
        if (NVRAM_SIMULATE_NEW_EEPROM)
            //...Can also range-check 'offset', so that the debug/test
            //  operation becomes limited to a specific NVRAM range.
        {
            // Simulate success of operation, instead of actually performing it.
            //  - CAUTION: ASSUMES no EEPROM pages remapped for wear-leveling.
            result = lengthWritten;
        }
        else
        {
            // Write/commit the affected pages to NVRAM.
            result = WriteAffectedPages(offset, lengthWritten);
        }
    }
    return result;
}

/**
* \author   Arto Kiremitdjian.
* \brief    Reads data from the section.
* \date     12/5/2011
* \param    sectOffset - Where to start reading: Offset relative to section.
* \param    pBuffer - Output buffer where to return the data that was read.
* \param    length - How many bytes to read, How much room in output buffer.
* \return   Number of bytes actually read.
* \retval   -1 = Failure.
*/
ssize_t Read (size_t const sectOffset, void* const pBuffer, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.

    // Verify parameter(s), except those that are only passed to callees.
    if (IsValidRange(sectOffset, length, true))
    {
        // Read desired NVRAM data.  NOTE: Result might not match 'length'.
        result = ReadShadow(sectOffset, pBuffer, length);
    }
    return result;
}

ssize_t PDSRead (size_t const logicalSectOffset, void* const pBuffer, size_t const length)
{
	ssize_t result = -1; // ASSUME failure.
	ssize_t bytesRead = 0;
	//calculate affected logical pages
	struct MemoryPageRangeInfo pageInfo;
	unsigned int i;

	if(!CalcAssociatedPages(logicalSectOffset,length,&pageInfo)) {
	    return -1;
	}

	for(i = 0; i < pageInfo.count; i++)
	{
		size_t length = 0;
		size_t sectOffset = 0;
		if(i == 0)
		{
			sectOffset = LogicalToPhysical(logicalSectOffset);
			length = pageInfo.firstLength;
		}
		else if(i == pageInfo.count-1)
		{
			sectOffset = LogicalToPhysical((pageInfo.firstIndex + i)*NVRAM_PAGE_SIZE);
			length = pageInfo.lastLength;
		}
		else
		{
			sectOffset = LogicalToPhysical((pageInfo.firstIndex + i)*NVRAM_PAGE_SIZE);
			length = NVRAM_PAGE_SIZE;
		}

		// Delegate to base-class version of same function.
		result = Read(sectOffset, (unsigned char*)pBuffer+bytesRead, length);
		if(result != -1)
		{
			bytesRead += result;
			result = bytesRead;
		}
		else
		{
			result = -1;
			break;
		}
	}

	//...Verify primary operation, which was to read data.
	if (-1 != result)
	{
		// Primary operation succeeded. Verify secondary one: Check CRC,
		//  but only if at least some of the data being read potentially
		//  contributed to it (Was the CRC calculated from such data?).
		if (isOutsideCrcRawData(logicalSectOffset, length) || isCrcValid)
		{
			// Secondary operation is either moot (not relevant to the CRC
			//  value at all) or successful (related to a valid CRC value).
			// Nothing to do: Success already based on primary operation.
		}
		else
		{
			// Use OS to report secondary operation's result: CRC error.
			//TODO        	errno = EACCES;
//            SetLastError(ERROR_CRC);

#if NVRDS_READ_FAIL_ON_CRC_ERR
			// CRC error makes data suspect: Avoid returning any to caller.
			//
			//  - NOTE: Already copied such data into the caller's buffer,
			//    which cannot be undone.  Now, just prepare to indicate a
			//    negative outcome through the function's return-code.
			//
			result = -1;
#endif
		}
	}
	else
	{
		// Nothing to do: Failure already based on primary operation.
	}
	return result;
}

/**
* \author   Arto Kiremitdjian.
* \date     1/10/2012
* \brief    (See comments of base-class implementation for all details.)
* \			With the exception that parameters are logical address rather than physical
*/
ssize_t PDSWrite (size_t const logicalSectOffset, const void* const pBuffer,
                                  size_t const logicalLength, bool const enforceWriteProtection)
{
	ssize_t result = -1; // ASSUME failure.
	ssize_t bytesWritten = 0;
	size_t sectOffset = 0;
	size_t length = 0;
	static unsigned int pageWriteCount = 0;
	unsigned int i;
	//calculate affected logical pages
	struct MemoryPageRangeInfo pageInfo;

//	mutex_lock(&nvramMutex);
//	printk("PDSWrite: %d %d\r\n", logicalSectOffset, logicalLength);
    if(!CalcAssociatedPages(logicalSectOffset,logicalLength,&pageInfo)) {
        return -1;
    }
	// Verify parameter(s) related to write-protected areas of the section.
	//  Allow writing under any of the following conditions:
	//  - Write-protection is temporarily disabled during this function call.
	//  - The affected write-area is not in any of the write-protected areas.
	//
	if ((!enforceWriteProtection) ||
		(isOutsideSignature(logicalSectOffset, logicalLength) &&
		isOutsideStoredCrc(logicalSectOffset, logicalLength) &&
		isOutsideWearTables(logicalSectOffset, logicalLength)))
	{
		for(i = 0; i < pageInfo.count; i++)
		{
			if(i == 0)
			{
				sectOffset = LogicalToPhysical(logicalSectOffset);
				length = pageInfo.firstLength;
			}
			else if(i == pageInfo.count-1)
			{
				sectOffset = LogicalToPhysical((pageInfo.firstIndex + i)*NVRAM_PAGE_SIZE);
				length = pageInfo.lastLength;
			}
			else
			{
				sectOffset = LogicalToPhysical((pageInfo.firstIndex + i)*NVRAM_PAGE_SIZE);
				length = NVRAM_PAGE_SIZE;
			}

			// Delegate to base-class version of same function.
			result = Write(sectOffset, (unsigned char*)pBuffer+bytesWritten, length, enforceWriteProtection);
			//if success, increment writes
			if(result != -1)
			{
				WearTables.TotalWrites[sectOffset/NVRAM_PAGE_SIZE]++;
				if((WearTables.TotalWrites[sectOffset/NVRAM_PAGE_SIZE] % WEAR_CHECK_INTERVAL) == 0)
				{
					swapFlags[sectOffset/NVRAM_PAGE_SIZE] = true;
					queue_work(wearLeveling_wq, &wearLeveling_work);
				}
				bytesWritten += result;
				result = bytesWritten;
			}
			else
			{
				result = -1;
				break;
			}
		}

		pageWriteCount += pageInfo.count;
		//see if we should update the wear tables
		if(pageWriteCount >= PAGE_WRITES_BEFORE_UPDATE)
		{
			SaveWearTables();
			pageWriteCount = 0;
		}

		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		// Check if any of the given data are relevant to the CRC value.

		if (isOutsideCrcRawData(logicalSectOffset, logicalLength))
		{
			// The given data would have no effect on the CRC value.
			// Nothing to do: Result already based on primary operation.
		}
		else
		{
			// At least some of the given data would affect the CRC value.
			//  Verify that such data were written, at least in part.
			if (-1 != result)
			{
				// Primary operation succeeded (actually, it did not fail).
				//  Some data affecting the CRC value might have changed.
				//  Prepare to update CRC: Verify that it is valid so far.
				if (isCrcValid)
				{
					// Update the stored CRC value.
					//
					//  - NOTE:  Result of the callee is ignored, because it
					//    would not affect the rest of the current function.
					//    The callee does at least the following upon error:
					//    Tracks the state of the CRC as invalid.
					//
					UpdateStoredCrc(); // CAUTION: Result ignored.
				}
				else
				{
					// Do not update invalid CRC, because that hides errors.
				}
			}
			else
			{
				// Primary operation failed, but it involved data that could
				//  affect the CRC value: Consider the CRC invalid.
				isCrcValid = false;
			}
			//...Done attempting to update the CRC. Check its state/outcome.
			// Use OS as secondary way to track/report any CRC-related errors.
			if (!isCrcValid)
			{
//TODO				SetLastError(ERROR_CRC);

#if NVRDS_WRITE_FAIL_ON_CRC_ERR
				// CRC error makes any stored data suspect.
				//
				//  - NOTE: Already attempted to store some data to persistent
				//    memory, which cannot be undone.  Now, just prepare to
				//    indicate negative outcome through function's return-code.
				//
				result = -1;
#endif
			}
		}
	}
	else
	{
		// Use OS to track/report secondary result: Write-protection error.
		//TODO SetLastError(ERROR_WRITE_PROTECT);

		printk("%s: ERROR: Not writable: offset %u, length %u, protected? %u.\r\n",
			__FUNCTION__, logicalSectOffset,
			logicalLength, enforceWriteProtection);

		// NOTE:  Write-protection errors do not affect/invalidate the CRC.
		//  Since the write-protection feature prevents an attempt to write
		//  any persistent data, it would leave the stored CRC value as-is.
	}

//	mutex_unlock(&nvramMutex);
    return result;
}

/**
* \author   Arto Kiremitdjian (based on recent s/w from Richard Ting).
* \date     1/30/2012
* \brief    Indicates whether a given write-area is away from the signature.
* \detail   Only tests the specified condition.  Does not write any data.
* \param    sectOffset - Where write-area starts: Offset relative to section.
* \param    length - How many bytes would be written.
* \return   Boolean: Whether function succeeded, Outcome of test condition.
*/
bool isOutsideSignature (size_t const sectOffset, size_t const length)
{
    bool result = false; // ASSUME negative outcome.

    // The offset right after the end of the given write-area.
    size_t const afterAffectedOffset = sectOffset + length;

    // The offset right after the end of the current write-protected area.
    size_t const afterWriteProtSpan = CS_UL_SIGNATURE + sizeof(SignatureType);

    // Verify that the given write-area does not overlap the current
    //  write-protected area.  Expect at least one of the following:
    //  - The given write-area starts after the write-protected area.
    //  - The given write-area ends before the write-protected area.
    if ((afterWriteProtSpan <= sectOffset) ||
        (CS_UL_SIGNATURE >= afterAffectedOffset))
    {
        result = true; // Prepare to report positive outcome.
    }
//    printk("%s: result %x\r\n", __FUNCTION__,result);
    return result;
}

/**
* \author   Arto Kiremitdjian (based on recent s/w from Richard Ting).
* \date     1/30/2012
* \brief    Indicates whether a given write-area is away from the CRC.
* \detail   Only tests the specified condition.  Does not write any data.
* \param    sectOffset - Where write-area starts: Offset relative to section.
* \param    length - How many bytes would be written.
* \return   Boolean: Whether function succeeded, Outcome of test condition.
*/
bool isOutsideStoredCrc (size_t const sectOffset, size_t const length)
{
    bool result = false; // ASSUME negative outcome.

    // The offset right after the end of the given write-area.
    size_t const afterAffectedOffset = sectOffset + length;

    // The offset right after the end of the current write-protected area.
    size_t const afterWriteProtSpan = CS_UL_SECTION3CRC + sizeof(CrcValueType);

    // Verify that the given write-area does not overlap the current
    //  write-protected area.  Expect at least one of the following:
    //  - The given write-area starts after the write-protected area.
    //  - The given write-area ends before the write-protected area.
    if ((afterWriteProtSpan <= sectOffset) ||
        (CS_UL_SECTION3CRC >= afterAffectedOffset))
    {
        result = true; // Prepare to report positive outcome.
    }
//    printk("%s: result %x\r\n", __FUNCTION__,result);
    return result;
}

/**
* \author   Arto Kiremitdjian.
* \date     1/31/2012
* \brief    Indicates whether a given write-area is away from the raw data
*           to which the CRC value applies (Such data affect the CRC value).
* \detail   Only tests the specified condition.  Does not write any data.
* \param    sectOffset - Where write-area starts: Offset relative to section.
* \param    length - How many bytes would be written.
* \return   Boolean: Whether function succeeded, Outcome of test condition.
*/
bool isOutsideCrcRawData (size_t const sectOffset, size_t const length)
{
    bool result = false; // ASSUME negative outcome.

    // The offset right after the end of the given write-area.
    //
    //  - CAUTION: Not necessarily a valid offset for reading/writing data.
    //             It is only for comparison with other offset values.
    //
    size_t const afterAffectedOffset = sectOffset + length;

    // The offset right after the end of the current special area.
    //
    //  - CAUTION: Not necessarily a valid offset for reading/writing data.
    //             It is only for comparison with other offset values.
    //
    size_t const afterSpecialSpan = SYSTEM_STORAGE + SYSTEM_STORAGE_SIZE;

    // Verify that the given write-area does not overlap the current
    //  special area.  Expect at least one of the following:
    //  - The given write-area starts after the special area.
    //  - The given write-area ends before the special area.
    if ((afterSpecialSpan <= sectOffset) ||
        (SYSTEM_STORAGE >= afterAffectedOffset))
    {
        result = true; // Prepare to report positive outcome.
    }
//    printk("%s: result %x\r\n", __FUNCTION__,result);
    return result;
}

bool isOutsideWearTables (size_t const sectOffset, size_t const length)
{
    bool result = false; // ASSUME negative outcome.

    // The offset right after the end of the given write-area.
    size_t const afterAffectedOffset = sectOffset + length;

    // The offset right after the end of the current write-protected area.
    size_t const afterWriteProtSpan = WEAR_LEVELING_TABLES + sizeof(WEAR_LEVELING_TABLES_STRUCT);

    // Verify that the given write-area does not overlap the current
    //  write-protected area.  Expect at least one of the following:
    //  - The given write-area starts after the write-protected area.
    //  - The given write-area ends before the write-protected area.
    if ((afterWriteProtSpan <= sectOffset) ||
        (WEAR_LEVELING_TABLES >= afterAffectedOffset))
    {
        result = true; // Prepare to report positive outcome.
    }
//    printk("%s: result %x\r\n", __FUNCTION__,result);
    return result;
}

/**
* \author   Arto Kiremitdjian
* \date     1/18/2012
* \brief    (See comments of base-class implementation for other details.)
* \detail   Performs the following operations:
*           - Clears whole persistent-memory section.
*           - Applies signature (in Data Store section 1 "Critical Storage").
*           - Invalidates the CRC (of Data Store section 3 "System Storage").
*/
bool Clear (void)
{
	bool result = false; // ASSUME failure.
	SignatureType const signatureNew = CS_SIGNATURE_VALUE;
	bool crcResult;

	mutex_lock(&nvramMutex);

    if (Fill(CRITICAL_STORAGE, 0, CRITICAL_STORAGE_SIZE) == CRITICAL_STORAGE_SIZE &&
		Fill(DRIVER_STORAGE, 0, NVRAM_CAPACITY-(CRITICAL_STORAGE_SIZE + WEAR_LEVELING_TABLES_SIZE)) == NVRAM_CAPACITY-(CRITICAL_STORAGE_SIZE + WEAR_LEVELING_TABLES_SIZE))
    {
        printk("%s: Cleared by base class.\r\n", __FUNCTION__);

        // Put desired signature at start of buffer.
        //
        //  - NOTE: Performed only AFTER clearing the section,
        //    so that the presence of a good signature in NVRAM
        //    indicates that section has been fully initialized.
        //
        //  - Not related to the CRC's state, because the data
        //    being written is not involved in the CRC calculation.
        //    Therefore, no need to use a base-class write function.
        //

        if (sizeof(signatureNew) == PDSWrite(
            CS_UL_SIGNATURE, &signatureNew, sizeof(signatureNew), false))
        {
            // Success so far: Data Store has just been initialized/cleared.
            printk("%s: Wrote signature.\r\n", __FUNCTION__);

            result = true; // Prepare to report success.
        }
        else
        {
            printk("%s: FAILED to write signature.\r\n", __FUNCTION__);
        }
    }
    else
    {
        printk("%s: FAILED base-class operation.\r\n", __FUNCTION__);
    }
    //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // The current function should clear/erase both the stored CRC value and
    //  the data to which it applies.  Even if the function fails, at least
    //  some of that might have been done.  Therefore, invalidate the CRC.
    //
    crcResult = InvalidateStoredCrc();

    // Allow any failure in recent step(s) to downgrade overall result.
    if (result)
    {
        result = crcResult;
    }
    mutex_unlock(&nvramMutex);
    return result;
}

/**
* \author   Rich Ting
* \date     2/26/2014
* \detail   Performs the following operations:
*           - Clears whole persistent-memory section.
*/
bool ClearAll (void)
{
	bool result = false; // ASSUME failure.
	mutex_lock(&nvramMutex);

    if (Fill(CRITICAL_STORAGE, 0, NVRAM_CAPACITY) == NVRAM_CAPACITY)
    {
        result = true;
    }
    else
    {
        printk("%s: FAILED\n", __FUNCTION__);
    }
    mutex_unlock(&nvramMutex);
    return result;
}

/**
* \author   Richard Ting, Arto Kiremitdjian.
* \date     1/26/2012
* \brief    (See comments of base-class implementation for all details.)
*/
bool ConsiderValid (void)
{
    bool result = false; // ASSUME failure.

    mutex_lock(&nvramMutex);
	// Perform corresponding operation in current class: Update CRC.
	if (UpdateStoredCrc())
	{
		result = true; // Prepare to report success.
	}
	else
	{
		printk("%s: FAILED to update CRC.\r\n", __FUNCTION__);
	}
	mutex_unlock(&nvramMutex);
    return result;
}

/**
* \author   Arto Kiremitdjian (based on recent s/w from Richard Ting).
* \date     1/30/2012
* \brief    Updates CRC (of Data Store section 3 "System Storage").
* \detail   Also, tracks the validity of the stored CRC value.
* \param    (None).
* \return   Boolean: Whether this function succeeded.
*/
bool UpdateStoredCrc (void)
{
    bool result = false; // ASSUME failure.

    CrcValueType crc = 0; // To be calculated.
    if (CalculateCrc(&crc))
    {
        // Calculated the CRC value: Store it in persistent memory.
        if (StoreAsCrc(crc))
        {
            result = true; // Prepare to report success.
        }
        else
        {
            printk("%s: FAILED to store CRC.\r\n", __FUNCTION__);
        }
    }
    else
    {
        printk("%s: FAILED to calculate CRC.\r\n", __FUNCTION__);
    }
    //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // The purpose of the current function is to update the stored CRC value.
    //  Therefore, the CRC's validity is tied to the outcome of the function.
    isCrcValid = result;

    // Use the OS as secondary way to track/report any CRC-related errors.
    if (!isCrcValid)
    {
//TODO    	errno = EACCES;
//        SetLastError(ERROR_CRC);
    }
    return result;
}

/**
* \author   Arto Kiremitdjian.
* \date     1/30/2012
* \brief    Invalidates CRC (of Data Store section 3 "System Storage").
* \detail   Also, tracks that the stored CRC value became invalid.
* \param    (None).
* \return   Boolean: Whether this function succeeded.
*/
bool InvalidateStoredCrc (void)
{
    bool result = false; // ASSUME failure.

    CrcValueType crc = 0; // To be calculated.
    if (CalculateCrc(&crc))
    {
        // Calculated the valid CRC value: Generate an invalid one from it.
        crc ^= ~0; // CAUTION: Makes CRC invalid.

        // Store the invalid CRC in persistent memory.
        if (StoreAsCrc(crc))
        {
            result = true; // Prepare to report success.
        }
        else
        {
            printk("%s: FAILED to store invalid CRC.\r\n", __FUNCTION__);
        }
    }
    else
    {
        printk("%s: FAILED to calculate CRC.\r\n", __FUNCTION__);
    }
    //- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // The purpose of the current function is to invalidate the stored CRC.
    //  Therefore, consider that CRC invalid whenever the function runs.
    isCrcValid = false;

    // Use the OS as secondary way to track/report intentional bad-CRC error.
    //TODO    errno = EACCES;
//    SetLastError(ERROR_CRC);

    return result;
}

/**
* \author   Arto Kiremitdjian.
* \date     1/30/2012
* \brief    Stores given CRC (for Data Store section 3 "System Storage").
* \note     Does not affect/track the state/validity of any CRC.
* \warning  Can store an invalid CRC value, if such a value is given.
* \param    crc - What value to store. WARNING: Does not validate it.
* \return   Boolean: Whether this function succeeded.
*/
bool StoreAsCrc (CrcValueType const crc)
{
	bool result = false; // ASSUME failure.

//	mutex_lock(&nvramMutex); done in higher level

    // Store the given CRC value in persistent memory.
    //
    //  - Not actually related to the CRC's state, because the data
    //    being written does not contribute to the CRC calculation.
    //    Therefore, no need to use a base-class write function.
    //
    if (sizeof(crc) == PDSWrite(CS_UL_SECTION3CRC, &crc, sizeof(crc), false))
    {
		result = true; // Prepare to report success.
    }
    else
    {
        printk("%s: FAILED to write CRC value.\r\n", __FUNCTION__);
    }
//    mutex_unlock(&nvramMutex);
    return result;
}

/**
* \author   Richard Ting, Arto Kiremitdjian.
* \date     1/26/2012
* \brief    Calculates CRC (of Data Store section 3 "System Storage").
* \note     Does not affect persistent memory, nor any CRC stored there.
* \param    crcOut - Output: Calculated value of CRC.
* \return   Boolean: Whether this function succeeded.
*/
bool CalculateCrc (CrcValueType* crcOut)
{
    bool result = false; // ASSUME failure.
    size_t const length = SYSTEM_STORAGE + SYSTEM_STORAGE_SIZE; // get entire NVRAM.
    size_t i;

    // Access the (read-only) data for which the CRC will be calculated.
    u8 const *const dataPtr = PeekShadow(0, length);
    if (NULL != dataPtr)
    {
		size_t PhysicalAddress = LogicalToPhysical(SYSTEM_STORAGE);
		//assumes start of CRC is on page boundary (for now)
        *crcOut = CalculateRunningCRC(dataPtr+PhysicalAddress, NVRAM_PAGE_SIZE, true);
		for(i = NVRAM_PAGE_SIZE; i < SYSTEM_STORAGE_SIZE; i+=NVRAM_PAGE_SIZE)
		{
			PhysicalAddress = LogicalToPhysical(SYSTEM_STORAGE + i);
			*crcOut = CalculateRunningCRC(dataPtr+PhysicalAddress, NVRAM_PAGE_SIZE, false);
		}
        result = true; // Prepare to report success.
    }
    else
    {
        printk("%s: FAILED to read data for CRC.\r\n", __FUNCTION__);
    }
    return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Checks if the wear tables is already initialized.
* \note     detected by (finding all 0's or all 1's in the table) or checking for a signature
* \return   Boolean: Whether this function succeeded.
*/
bool IsWearTablesInitialized()
{
#if 0
	//assumes true, if it finds a value that is not, then set false
	bool allZeros = true;
	bool allFs = true;
	unsigned char* pBuffer = (unsigned char*)&WearTables;
	for(int i = 0; i < WEAR_LEVELING_TABLES_SIZE; i++)
	{
		if(pBuffer[i] != 0x00)
		{
			allZeros = false;
		}
		if(pBuffer[i] != 0xFF)
		{
			allFs = false;
		}
	}
	return !allZeros && !allFs;
#else
	return (WearTables.Signature == WL_SIGNATURE_VALUE);
#endif
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Initialized wear tables to default values.
* \note     default values based on 1 to 1 mapping of logical to physical address, and no writes to any page.
* \return   Boolean: Whether this function succeeded.
*/
bool InitializeWearTables()
{
	bool result = false;
	size_t i;
	mutex_lock(&nvramMutex);
	for(i = 0; i < WL_WEAR_TABLE_NUMBER_ENTRIES; i++)
	{
		//default map, logical page = physical page
		WearTables.PageAllocationTable[i] = i;
	}
	for(i = 0; i < WL_WEAR_TABLE_NUMBER_ENTRIES; i++)
	{
		//initialize all counters to 0
		WearTables.TotalWrites[i] = 0;
	}
	WearTables.Signature = WL_SIGNATURE_VALUE;
	if(Write(WEAR_LEVELING_TABLES, &WearTables, sizeof(WearTables),false) == sizeof(WearTables))
	{
		printk("%s: Success.\r\n", __FUNCTION__);
		for(i = WEAR_LEVELING_TABLES/NVRAM_PAGE_SIZE; i < (WEAR_LEVELING_TABLES+sizeof(WearTables))/NVRAM_PAGE_SIZE; i++)
		{
			WearTables.TotalWrites[i]++;
		}
		result = true;
	}
	else
	{
		printk("%s: FAILED.\r\n", __FUNCTION__);
		result = false;
	}
	mutex_unlock(&nvramMutex);
	return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Stores a local copy of the wear tables from shadow ram.
* \return   Boolean: Whether this function succeeded.
*/
bool LoadWearTables()
{
	bool result = false;
	//use base class type since we need physical location and wont be moved.
	if(Read(WEAR_LEVELING_TABLES, &WearTables, sizeof(WearTables)) == sizeof(WearTables))
	{
		printk("%s: Success.\r\n", __FUNCTION__);
		result = true;
	}
	else
	{
		printk("%s: FAILED.\r\n", __FUNCTION__);
		result = false;
	}

	return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Checks for incomplete swap error in the wear tables, and corrects if it finds one.
* \return   Boolean: Whether this function succeeded.
*/
bool ValidateWearTables()
{
	bool result = false;
	int i;

	mutex_lock(&nvramMutex);
	//check if there was a page swap that did not previously finish
	if(WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE] != WL_SWAP_PAGE/NVRAM_PAGE_SIZE)
	{
		printk("%s: Wear Table Error - incomplete write operation.\r\n", __FUNCTION__);
		//find page where swap data came from or about to go to.
		for(i = 0; i < WL_WEAR_TABLE_NUMBER_ENTRIES; i++)
		{
			if(WearTables.PageAllocationTable[i] == WL_SWAP_PAGE/NVRAM_PAGE_SIZE)
			{
				if(Write(i*NVRAM_PAGE_SIZE, &WearTables.SwapPage, WL_SWAP_PAGE_SIZE, false) == WL_SWAP_PAGE_SIZE)
				{
					printk("%s: Success.\r\n", __FUNCTION__);
					WearTables.PageAllocationTable[i] = WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE];
					WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE] = WL_SWAP_PAGE/NVRAM_PAGE_SIZE;
					WearTables.TotalWrites[i]++;
					result = true;
				}
				else
				{
					printk("%s: FAILED.\r\n", __FUNCTION__);
					mutex_unlock(&nvramMutex);
					return false;
				}

				if(Write(WL_PAGE_ALLOCATION_TABLE, &WearTables.PageAllocationTable, sizeof(WearTables.PageAllocationTable), false) == sizeof(WearTables.PageAllocationTable))
				{
					printk("%s: Incomplete write error correction success.\r\n", __FUNCTION__);
					for(i = WL_PAGE_ALLOCATION_TABLE/NVRAM_PAGE_SIZE; i < (WL_PAGE_ALLOCATION_TABLE+sizeof(WearTables.PageAllocationTable))/NVRAM_PAGE_SIZE; i++)
					{
						WearTables.TotalWrites[i]++;
					}
					result = true;
				}
				else
				{
					printk("%s: FAILED.\r\n", __FUNCTION__);
					mutex_unlock(&nvramMutex);
					return false;
				}

				//once done, we can exit loop
				break;
			}
			else
			{
				printk("%s incomplete pages not found - reinitialze wear tables\n",__FUNCTION__);
				mutex_unlock(&nvramMutex);
				if(!InitializeWearTables())
				{
					printk("%s: FAILED to initialize wear tables.\r\n", __FUNCTION__ );
					result = true;
					mutex_lock(&nvramMutex);
					break;
				}
				mutex_lock(&nvramMutex);
			}
		}
	}
	else
	{
		result = true;
	}
	mutex_unlock(&nvramMutex);
	return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Saves all the wear tables into nvram.
* \return   Boolean: Whether this function succeeded.
*/
bool SaveWearTables()
{
	bool result = false;
	int i;

//	mutex_lock(&nvramMutex); done in higher layer

	//use base class type since we need physical location and wont be moved.
	if(Write(WEAR_LEVELING_TABLES, &WearTables, sizeof(WearTables), false) == sizeof(WearTables))
	{
		printk("%s: Success.\r\n", __FUNCTION__);
		for(i = WEAR_LEVELING_TABLES/NVRAM_PAGE_SIZE; i < (WEAR_LEVELING_TABLES+sizeof(WearTables))/NVRAM_PAGE_SIZE; i++)
		{
			WearTables.TotalWrites[i]++;
		}
		result = true;
	}
	else
	{
		printk("%s: FAILED.\r\n", __FUNCTION__);
		result = false;
	}
//	mutex_unlock(&nvramMutex); done in higher layer
	return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Swaps the contents of 2 physical pages.
* \warning  If fails, can put file table in inconsistant state.  Must run ValidateWearTables to fix.
* \param    PhysicalPage_1 - Index of first page to swap.
* \param    PhysicalPage_2 - Index of second page to swap.
* \return   Boolean: Whether this function succeeded.
*/
bool Swap(size_t PhysicalPage_1, size_t PhysicalPage_2)
{
	bool result = true;
	int i;
	size_t physicalAddress_1, physicalAddress_2;

	printk("%s: Pages: %i %i.\r\n", __FUNCTION__,PhysicalPage_1,PhysicalPage_2);

	//range check index just in case.
	if(PhysicalPage_2 < 0 || PhysicalPage_2 >= WL_WEAR_TABLE_NUMBER_ENTRIES ||
		PhysicalPage_1 < 0 || PhysicalPage_1 >= WL_WEAR_TABLE_NUMBER_ENTRIES)
	{
		printk("%s: Error: Invalid Index\r\n", __FUNCTION__);
		return false;
	}

	//currently is the min, dont swap
	if(PhysicalPage_2 == PhysicalPage_1)
	{
		printk("%s: Same Page: no need to swap.\r\n", __FUNCTION__);
		return true;
	}
	physicalAddress_1 = PhysicalPage_1*NVRAM_PAGE_SIZE;
	physicalAddress_2 = PhysicalPage_2*NVRAM_PAGE_SIZE;
//	mutex_lock(&nvramMutex); done in upper layer

	//read out the max page from RAM that is going to be written
	if(Read(physicalAddress_1, WearTables.SwapPage, WL_SWAP_PAGE_SIZE) == NVRAM_PAGE_SIZE)
	{
	}
	else
	{
		printk("%s: FAILED 1.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	//Write max page to swap space
	if(Write(WL_SWAP_PAGE, WearTables.SwapPage, WL_SWAP_PAGE_SIZE, false) == NVRAM_PAGE_SIZE)
	{
		//change table to point to swap page
		WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE] = WearTables.PageAllocationTable[PhysicalPage_1];
		WearTables.PageAllocationTable[PhysicalPage_1] = WL_SWAP_PAGE/NVRAM_PAGE_SIZE;
		//increment wear tables count.  not sure if this occurs often enough to keep track
		WearTables.TotalWrites[WL_SWAP_PAGE/NVRAM_PAGE_SIZE]++;
	}
	else
	{
		printk("%s: FAILED 2.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	//update page allocation tables
	if(Write(WL_PAGE_ALLOCATION_TABLE, &WearTables.PageAllocationTable, sizeof(WearTables.PageAllocationTable), false) == sizeof(WearTables.PageAllocationTable))
	{
		for(i = WL_PAGE_ALLOCATION_TABLE/NVRAM_PAGE_SIZE; i < (WL_PAGE_ALLOCATION_TABLE+sizeof(WearTables.PageAllocationTable))/NVRAM_PAGE_SIZE; i++)
		{
			WearTables.TotalWrites[i]++;
		}
	}
	else
	{
		printk("%s: FAILED 3.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}

	//read out the min page from RAM that is going to be written
	if(Read(physicalAddress_2, WearTables.SwapPage, WL_SWAP_PAGE_SIZE) == NVRAM_PAGE_SIZE)
	{
	}
	else
	{
		printk("%s: FAILED 4.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}

	//Write min page to max page
	if(Write(physicalAddress_1, WearTables.SwapPage, WL_SWAP_PAGE_SIZE, false) == NVRAM_PAGE_SIZE)
	{
		unsigned short tempPageIndex = WearTables.PageAllocationTable[PhysicalPage_1];
		//change table to point to max page
		WearTables.PageAllocationTable[PhysicalPage_1] = WearTables.PageAllocationTable[PhysicalPage_2];
		WearTables.PageAllocationTable[PhysicalPage_2] = tempPageIndex;
		//increment wear tables count.  not sure if this occurs often enough to keep track
		WearTables.TotalWrites[PhysicalPage_1]++;
	}
	else
	{
		printk("%s: FAILED 5.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	//update page allocation tables and total counters table (validation after this step will finish off swap rather than undo)
	if(Write(WL_PAGE_ALLOCATION_TABLE, &WearTables.PageAllocationTable, sizeof(WearTables.PageAllocationTable)+sizeof(WearTables.TotalWrites), false) == sizeof(WearTables.PageAllocationTable)+sizeof(WearTables.TotalWrites))
	{
		for(i = WEAR_LEVELING_TABLES/NVRAM_PAGE_SIZE; i < (WEAR_LEVELING_TABLES+sizeof(WearTables))/NVRAM_PAGE_SIZE; i++)
		{
			WearTables.TotalWrites[i]++;
		}
	}
	else
	{
		printk("%s: FAILED 6.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}

	//read out the swap page from RAM that is going to be written
	if(Read(WL_SWAP_PAGE, WearTables.SwapPage, WL_SWAP_PAGE_SIZE) == NVRAM_PAGE_SIZE)
	{
	}
	else
	{
		printk("%s: FAILED 7.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	//Write swap page to min page
	if(Write(physicalAddress_2, WearTables.SwapPage, WL_SWAP_PAGE_SIZE, false) == NVRAM_PAGE_SIZE)
	{
		//change table to point to min page
		WearTables.PageAllocationTable[PhysicalPage_2] = WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE];
		WearTables.PageAllocationTable[WL_SWAP_PAGE/NVRAM_PAGE_SIZE] = WL_SWAP_PAGE/NVRAM_PAGE_SIZE;
		//increment wear tables count.  not sure if this occurs often enough to keep track
		WearTables.TotalWrites[PhysicalPage_2]++;
	}
	else
	{
		printk("%s: FAILED 8.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	//update page allocation tables
	if(Write(WL_PAGE_ALLOCATION_TABLE, &WearTables.PageAllocationTable, sizeof(WearTables.PageAllocationTable), false) == sizeof(WearTables.PageAllocationTable))
	{
		for(i = WL_PAGE_ALLOCATION_TABLE/NVRAM_PAGE_SIZE; i < (WL_PAGE_ALLOCATION_TABLE+sizeof(WearTables.PageAllocationTable))/NVRAM_PAGE_SIZE; i++)
		{
			WearTables.TotalWrites[i]++;
		}
	}
	else
	{
		printk("%s: FAILED 9.\r\n", __FUNCTION__);
//		mutex_unlock(&nvramMutex);
		return false;
	}
	LogicalToPhysicalTable[WearTables.PageAllocationTable[PhysicalPage_1]] = PhysicalPage_1;
	LogicalToPhysicalTable[WearTables.PageAllocationTable[PhysicalPage_2]] = PhysicalPage_2;
	//swap succeeded
	printk("%s: Swap succeeded.\r\n", __FUNCTION__);
//	mutex_unlock(&nvramMutex);
	return result;
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Converts a Physical Address to a Logical Address.
* \param    PhysicalAddress - The Physical address to convert from.
* \return   size_t: Logical Address corresponding to the Physical address.
*/
size_t PhysicalToLogical(size_t const PhysicalAddress)
{
	//Logical addresses page size = 256
	size_t PhysicalPage = PhysicalAddress/NVRAM_PAGE_SIZE;
	size_t LogicalPage = WearTables.PageAllocationTable[PhysicalPage];
//	printk("%s: PhysicalAddress %x mapped to %x\r\n", __FUNCTION__, PhysicalAddress,(LogicalPage << 8) | (PhysicalAddress & 0xFF));
	return (LogicalPage * NVRAM_PAGE_SIZE) | (PhysicalAddress % NVRAM_PAGE_SIZE);
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Generates a table to do Logical to physical address mapping.
* \warning  based off of Page allocation table.  If multiple physical pages map to same logical page, will use last physical page found.
*/
void CreateLogicalToPhysicalTable(void)
{
	size_t PhysicalPage;
	for(PhysicalPage = 0; PhysicalPage < WL_WEAR_TABLE_NUMBER_ENTRIES; PhysicalPage++)
	{
		size_t LogicalPage = WearTables.PageAllocationTable[PhysicalPage];
		LogicalToPhysicalTable[LogicalPage] = PhysicalPage;
//		printk("%s: LogicalPage %x mapped to %x\r\n", __FUNCTION__, LogicalPage,PhysicalPage);
	}
}

/**
* \author   Richard Ting
* \date     2/24/2012
* \brief    Converts a Logical Address to a Physical Address.
* \param    LogicalAddress - The logical address to convert from.
* \return   size_t: Physical Address corresponding to the logical address.
*/
size_t LogicalToPhysical(size_t const LogicalAddress)
{
	//Logical addresses page size = 256
	size_t LogicalPage = LogicalAddress/NVRAM_PAGE_SIZE;
	size_t PhysicalPage = LogicalToPhysicalTable[LogicalPage];
//	printk("%s: LogicalAddress %x mapped to %x\r\n", __FUNCTION__, LogicalAddress,(PhysicalPage << 8) | (LogicalAddress & 0xFF));
	return (PhysicalPage * NVRAM_PAGE_SIZE) | (LogicalAddress % NVRAM_PAGE_SIZE);
}

static loff_t nvram_llseek(struct file *file, loff_t offset, int origin)
{
//	printk(KERN_ERR "nvram_llseek\n");
	wait_for_completion(&init_completed);

	switch (origin) {
	case 1:
		offset += file->f_pos;
		break;
	case 2:
		offset += nvram_len;
		break;
	}
	if (offset < 0)
		return -EINVAL;

	file->f_pos = offset;

	return file->f_pos;
}

/*
 * Corrects count variable.
 * Returns 0 for success, non-zero for error.
 */
static ssize_t pre_access(size_t *count, loff_t *ppos)
{
    if(*ppos >= nvram_len)
        return -1; // Signal EOF
    if(*ppos + (*count) > nvram_len)
        (*count) = nvram_len - *ppos;
    return 0;
}

/*
 * Repositions the file offset after access.
 */
static ssize_t post_access(ssize_t result, loff_t *ppos)
{
    if(result < 0)
        return -EFAULT;

    (*ppos) += result;
    return result;
}

static ssize_t __read_nvram_locked(char *buf, size_t count, loff_t *ppos)
{
    ssize_t result;
    if(pre_access(&count, ppos)) {
        /* pre_access returns non-zero for error. Change to 0 for EOF here. */
        return 0;
    }
    result = PDSRead ((size_t)(*ppos), buf, count);
    return post_access(result, ppos);
}

ssize_t m24m01_read(char *buf, size_t count, loff_t pos)
{
    ssize_t result;
    wait_for_completion(&init_completed);
    mutex_lock(&nvramMutex);
    result = __read_nvram_locked(buf, count, &pos);
    mutex_unlock(&nvramMutex);
    return result;
}
EXPORT_SYMBOL(m24m01_read);

static ssize_t sysfs_read_nvram(struct file *file, char __user *buf,
              size_t count, loff_t *ppos)
{
    ssize_t result;
    wait_for_completion(&init_completed);
    mutex_lock(&nvramMutex);
    result = __read_nvram_locked(gEEprom->usrBuf, min(count, PAGE_SIZE), ppos);
    if(result < 0)
    {
    	mutex_unlock(&nvramMutex);
    	return -EFAULT;
    }
    result -= copy_to_user(buf, gEEprom->usrBuf, result);
    mutex_unlock(&nvramMutex);
    return result;
}

static ssize_t __write_nvram_locked(const char *buf, size_t count, loff_t *ppos)
{
    ssize_t result;
    if(pre_access(&count, ppos)) {
        /* pre_access returns non-zero for error. Change to 0 for EOF here. */
        return 0;
    }
    result = PDSWrite ((size_t)(*ppos), buf, count, true);
    return post_access(result, ppos);
}

ssize_t m24m01_write(char *buf, size_t count, loff_t pos)
{
    ssize_t result;
    wait_for_completion(&init_completed);
    mutex_lock(&nvramMutex);
    result = __write_nvram_locked(buf, count, &pos);
    mutex_unlock(&nvramMutex);
    return result;
}
EXPORT_SYMBOL(m24m01_write);

static ssize_t sysfs_write_nvram(struct file *file, const char __user *buf,
			   size_t count, loff_t *ppos)
{
    ssize_t result;
    wait_for_completion(&init_completed);
    mutex_lock(&nvramMutex);

    result = copy_from_user(gEEprom->usrBuf, buf, min(count, PAGE_SIZE));
    count -= result; //result is how many bytes were not copied
    result = __write_nvram_locked(gEEprom->usrBuf, count, ppos);
    mutex_unlock(&nvramMutex);
    return result;
}

static int nvram_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int result = -1; // ASSUME failure.
//	printk(KERN_ERR "nvram_ioctl %x %u\n",cmd,arg);
	switch(cmd) {
#ifdef CONFIG_PPC_PMAC
	case OBSOLETE_PMAC_NVRAM_GET_OFFSET:
		printk(KERN_WARNING "nvram: Using obsolete PMAC_NVRAM_GET_OFFSET ioctl\n");
	case IOC_NVRAM_GET_OFFSET: {
		int part, offset;

		if (!machine_is(powermac))
			return -EINVAL;
		if (copy_from_user(&part, (void __user*)arg, sizeof(part)) != 0)
			return -EFAULT;
		if (part < pmac_nvram_OF || part > pmac_nvram_NR)
			return -EINVAL;
		offset = pmac_get_partition(part);
		if (copy_to_user((void __user*)arg, &offset, sizeof(offset)) != 0)
			return -EFAULT;
		break;
	}
#endif /* CONFIG_PPC_PMAC */
//	case IOC_NVRAM_SYNC:
//		nvram_sync();
//		break;
	case IOC_PDS_CLEAR:
		// Verify precondition(s).
		if (file->f_mode & FMODE_WRITE)
		{
			if (Clear())
			{
				result = 0; // Prepare to report success.
			}
		}
		break;
	case IOC_PDS_CLEARALL:
		// Verify precondition(s).
		if (file->f_mode & FMODE_WRITE)
		{
			if (ClearAll())
			{
				result = 0; // Prepare to report success.
			}
		}
		break;
	case IOC_PDS_REWRITE_CRC:
		// Verify precondition(s).
		if (file->f_mode & FMODE_WRITE)
		{
			if (ConsiderValid())
			{
				result = 0; // Prepare to report success.
			}
		}
		break;
	case IOC_PDS_IS_CRC_VALID:
		// Verify precondition(s).
		if (file->f_mode & FMODE_READ)
		{
			if (copy_to_user((void __user*)arg, &isCrcValid, sizeof(isCrcValid)) != 0)
				return -EFAULT;
			else
				result = 0; // Prepare to report success.

		}
		break;
	default:
		return -EINVAL;
	}

	return result;
}

static long nvram_unlocked_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret;

	wait_for_completion(&init_completed);

	mutex_lock(&fs_mutex);

	ret = nvram_ioctl(file, cmd, arg);

	mutex_unlock(&fs_mutex);

	return ret;
}

const struct file_operations nvram_fops = {
	.owner		= THIS_MODULE,
	.llseek		= nvram_llseek,
	.read		= sysfs_read_nvram,
	.write		= sysfs_write_nvram,
	.unlocked_ioctl	= nvram_unlocked_ioctl,
};

static struct miscdevice nvram_dev = {
	.minor 		= NVRAM_MINOR,
	.name  		= DRIVER_NAME,
//	.nodename	= "pds",
	.fops  		= &nvram_fops
};

static int m24m01_initialize(struct i2c_client *client)
{
	bool result = false; // ASSUME failure.
	char  threadName[]="WearLevelingThread";
	SignatureType signatureOld = 0;
	CrcValueType storedCrc; // To read stored value.
	CrcValueType expectedCrc; // To calculate.
	size_t i;

	    // Initialize the CRC table (maintained in another s/w module).
	    InitCRCLookupTable(); // TODO: Move to better place, Only invoke once.

	//wont actually loop, has a break at the end.  Used to exit early
	for(;;){
		mutex_init(&fs_mutex);
		mutex_init(&nvramMutex);
		//load tables
		if(!LoadWearTables())
		{
			printk("%s: FAILED to load wear tables.\r\n", __FUNCTION__ );
			break;
		}

		//check if they are initialized
		if(!IsWearTablesInitialized())
		{
			printk("%s: wear tables not initialized.\r\n", __FUNCTION__ );
			//initialize if they werent already initialized
			if(!InitializeWearTables())
			{
				printk("%s: FAILED to initialize wear tables.\r\n", __FUNCTION__ );
				break;
			}
		}

		//create mapping table
		CreateLogicalToPhysicalTable();

		if(!ValidateWearTables())
		{
			printk("%s: FAILED to validate wear tables.\r\n", __FUNCTION__);
			break;
		}

		wearLeveling_wq = create_singlethread_workqueue(threadName);
		if (NULL == wearLeveling_wq)
		{
			printk(KERN_ERR "%s() - ERROR: Could not create the Work Queue due to insufficient memory.\n", __FUNCTION__);
			break;
		}

		INIT_WORK(&wearLeveling_work, WearLevelingThread);


		// Read the existing Data Store signature.
		//
		//  - The CRC's state should have no effect, because the data
		//    being read was not involved in the CRC calculation.
		//    Therefore, no need to use a base-class read function.
		//

		if (sizeof(signatureOld) != PDSRead(CS_UL_SIGNATURE, &signatureOld, sizeof(signatureOld)))
		{
			printk("%s: FAILED to read signature.\r\n", __FUNCTION__ );
			break;
		}
		// Verify the Data Store signature.
		if (CS_SIGNATURE_VALUE != signatureOld)
		{
			// Signature does not match: Initialize the Data Store section.
			//  - Does not necessarily hurt outcome of current function,
			//    but indicates the total loss of any data the user had
			//    previously stored: Such data will be cleared/erased.
			printk("%s: WARNING: Signature %08Xh, but expected %08Xh. Initializing.\r\n",
				__FUNCTION__, signatureOld,
				CS_SIGNATURE_VALUE);

			result = Clear(); // Clear the associated section.
			break;
		}
		// Success so far: Data Store already seems initialized.
		printk("%s: Signature %08Xh = Expected value %08Xh.\r\n",
			__FUNCTION__, signatureOld,
			CS_SIGNATURE_VALUE );

		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - -
		// Verify whatever CRC value had already been stored.

		isCrcValid = false; // ASSUME that stored CRC value is bad.

		// Determine the expected CRC value.

		if (!CalculateCrc(&expectedCrc))
		{
			printk("%s: FAILED to calculate expected CRC.\r\n", __FUNCTION__ );
			break;
		}
		// Obtain whatever CRC value was already stored.
		//
					//  - The CRC's state should have no effect, because the data
					//    being read was not involved in the CRC calculation.
					//    Therefore, no need to use a base-class read function.
		//

		if (sizeof(storedCrc) != PDSRead(
			CS_UL_SECTION3CRC, &storedCrc, sizeof(storedCrc)))
		{
			printk("%s: FAILED to read stored CRC.\r\n",__FUNCTION__);
			break;
		}
		// Enough steps have been completed to consider the
		//  current function successful:  The memory section
		//  was already initialized (based on the signature),
		//  and CRC values were read and calculated.
		//
		//  NOTE:  The validity of the CRC (of Data Store
		//  section 3 "System Storage") does not affect the
		//  initialization status of the whole Data Store.
		//
		result = true; // Prepare to report success.

		// Verify that expected and stored CRC values match.
		if (expectedCrc == storedCrc)
		{
			isCrcValid = true; // Stored CRC is already valid.
		}
		else
		{
			// CRC values do not match.
			//  - Does not hurt outcome of current function,
			//    but predicts possible "bad CRC" errors in
			//    read/write operations that occur later on.
			printk("%s: WARNING: CRC %08Xh, but expected %08Xh.\r\n",
				__FUNCTION__, storedCrc,
				expectedCrc );
		}

		//rescan for any pages that need swapping (if validate undid some changes)
		for(i = 0; i < NVRAM_CAPACITY/NVRAM_PAGE_SIZE; i++)
		{
			if(((WearTables.TotalWrites[i] % WEAR_CHECK_INTERVAL) == 0) && (WearTables.TotalWrites[i] != 0))
			{
				swapFlags[i/NVRAM_PAGE_SIZE] = true;
				queue_work(wearLeveling_wq, &wearLeveling_work);
			}
		}
		break;
	}
	// Clean-up if any part of the initialization attempt failed.
	if (!result)
	{
		if (NULL != wearLeveling_wq)
			destroy_workqueue(wearLeveling_wq);
//TODO	        CleanUp();
	}
	return result;


	return 0;
}

static int init_thread(void * _client)
{
	struct i2c_client *client = _client;
	// Initially fill shadow RAM with data from NVRAM h/w.
	if (NVRAM_CAPACITY == ReadAffectedPages(0, NVRAM_CAPACITY))
	{

	}

	/* Initialize m24m01 */
	m24m01_initialize(client);

	complete_all(&init_completed);
	return 0;
}

static int __devinit m24m01_probe(struct i2c_client *client,
					const struct i2c_device_id *id)
{
	struct m24m01_eeprom *eeprom;
	struct miscdevice *misc;
	int error;
	struct task_struct * init_task;

	// Initialize the event timer.
//	if (eventTimer.Init())
	// just use sleeps?

	printk(KERN_ERR "m24m01_probe\n");

	misc = &nvram_dev;

	eeprom = kzalloc(sizeof(struct m24m01_eeprom), GFP_KERNEL);
	eeprom->client = client;
	eeprom->misc = misc;
	gEEprom = eeprom;
	i2c_set_clientdata(client, eeprom);
	// Allocate shadow RAM to reduce reading from h/w.
	shadowPtr = kzalloc(NVRAM_CAPACITY, GFP_KERNEL);



	if (!shadowPtr) {
		dev_err(&client->dev, "failed to allocate memory\n");
		error = -ENOMEM;
		goto failed_free_mem;
	}

	printk(KERN_INFO "m24m01 misc %d %d %d\n",misc->minor,nvram_dev.minor,MISC_DYNAMIC_MINOR);

//	device_init_wakeup(&client->dev, 1);

	init_task = kthread_run(init_thread, client, "EEProm_Init_Thread");
	if (IS_ERR(init_task)) {
		error = PTR_ERR(init_task);
		printk("%s: kthread error %x\n",__FUNCTION__,error);
		goto failed_free_mem;
	}
	return 0;

failed_free_mem:
//	input_free_device(misc);
	kfree(eeprom);
	return error;
}

static int __devexit m24m01_remove(struct i2c_client *client)
{
	struct m24m01_eeprom *eeprom = i2c_get_clientdata(client);

    /* Start cleaning up by removing any delayed work and the timer */
    if (cancel_delayed_work_sync((struct delayed_work *)&wearLeveling_work) < 0)
    {
        printk(KERN_ERR "%s() - ERROR: Could not remove all work from the Work Queue.\n", __FUNCTION__);
    }

	misc_deregister(eeprom->misc);
	kfree(eeprom);
	gEEprom = NULL;

	return 0;
}

#define m24m01_suspend	NULL
#define m24m01_resume	NULL

static const struct i2c_device_id m24m01_ids[] = {
	{ "m24m01", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, m24m01_ids);

static struct i2c_driver m24m01_i2c_driver = {
	.driver = {
		.name = "m24m01",
	},
	.probe		= m24m01_probe,
	.remove		= __devexit_p(m24m01_remove),
	.suspend	= m24m01_suspend,
	.resume		= m24m01_resume,
	.id_table	= m24m01_ids,
};

static int __init m24m01_init(void)
{
	int ret = 0;

	printk(KERN_INFO "m24m01 non-volatile memory driver\n");
	ret = misc_register(&nvram_dev);

	if (ret != 0)
	{
		printk("failed to register m24m01 device %d\n",ret);
		goto out;
	}

	nvram_len = NVRAM_CAPACITY;

out:

	return i2c_add_driver(&m24m01_i2c_driver);

}

static void __exit m24m01_cleanup_module(void)
{
	if (NULL != wearLeveling_wq)
		destroy_workqueue(wearLeveling_wq);

	misc_deregister(&nvram_dev);
	i2c_del_driver(&m24m01_i2c_driver);
}

module_init(m24m01_init);
module_exit(m24m01_cleanup_module);
